#!/usr/bin/perl

# This file is part of Koha.
#
# Koha is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Koha is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Koha; if not, see <http://www.gnu.org/licenses>.

use Modern::Perl;

use Getopt::Long qw( GetOptions );
use Pod::Usage qw( pod2usage );

use Koha::Script -cron;

use C4::Context;
use Koha::Items;

my $dbh = C4::Context->dbh();

my $query = {
    target_items => q|SELECT itemnumber, biblionumber from items|
};

my $GLOBAL = {
      query => $query
    , sth   => {}
};

my $OPTIONS = {
      where => []
    , flags    => {
            verbose   => ''
          , commit    => ''
          , help      => ''
          , manual    => ''
          , version   => ''
      }
};

GetOptions(
      'where=s'    => $OPTIONS->{where}
    , 'v|verbose'  => sub { $OPTIONS->{flags}->{verbose}   = 1 }
    , 'V|version'  => sub { $OPTIONS->{flags}->{version}   = 1 }
    , 'h|help'     => sub { $OPTIONS->{flags}->{help}      = 1 }
    , 'm|manual'   => sub { $OPTIONS->{flags}->{manual}    = 1 }
    , 'c|commit'   => sub { $OPTIONS->{flags}->{commit}    = 1 } # aka DO-EET!
);

my @where = @{ $OPTIONS->{where} };

pod2usage( -verbose => 2 ) if  $OPTIONS->{flags}->{manual};
pod2usage( -verbose => 1 ) if  $OPTIONS->{flags}->{help};
pod2usage( -verbose => 1 -msg => 'You must supply at least one --where option' ) if scalar @where == 0;

sub verbose {
    say @_ if $OPTIONS->{flags}->{verbose};
}

my $where_clause = ' where ' . join ( " and ", @where );

verbose "Where statement: $where_clause";

# FIXME Use Koha::Items instead
$GLOBAL->{sth}->{target_items} = $dbh->prepare( $query->{target_items} . $where_clause  );
$GLOBAL->{sth}->{target_items}->execute();

DELITEM: while ( my $item = $GLOBAL->{sth}->{target_items}->fetchrow_hashref() ) {

    my $item_object = Koha::Items->find($item->{itemnumber});
    my $safe_to_delete = $item_object->safe_to_delete;
    if( $safe_to_delete )  {
        $item_object->safe_delete
            if $OPTIONS->{flags}->{commit};
        verbose "Deleting '$item->{itemnumber}'";
    } else {
        verbose sprintf "Item '%s' not deleted: %s", $item->{itemnumber}, @{$safe_to_delete->messages}[0]->message
    }
}

=head1 NAME

delete_items.pl - A batch item deletion tool, which generates a query against the items database and deletes the items matching the criteria specified in the command line arguments.

=head1 SYNOPSIS

delete_items.pl [--help|--manual]

delete_items.pl [--verbose] --where "I<SQL CONDITIONAL EXPRESSION>" ... [--commit]

=cut

=head1 OPTIONS

=over 8

=item B<--help>

Show the brief help information.

=item B<--manual>

Read the manual, with examples.

=item B<--verbose>

Send the "WHERE" clause generated by the collected C<--where>
arguments, as well as items affected to Standard Out.

=item B<--where>

The C<--where> option may called multiple times. The following argument
must be a syntactically valid SQL statement which is part of the C<WHERE>
clause querying the items table. These are joined by C<AND>.

=item B<--commit>

No items will be deleted unless the C<--commit> flag is present.

=back

=cut


=head1 EXAMPLES

  The following is an example of this script:

 delete_items.pl --where "items.withdrawn ! 0"  --where "items.withdrawn_on < $(date --date="13 month ago" --rfc-3339=date)" --commit

 delete_items.pl --where "itemlost >= '1'" --where "itemlost <='4'" --where "itemlost_on < '2014-04-28'" --commit

=cut


=head1 DESCRIPTION

 This is a lightweight batch deletion tool for items, suitable for running in a cron job.

=cut


=head1 AUTHOR

 Barton Chittenden <barton@bywatersolutions.com>

=cut
